// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_IAUDIO_SINK_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_IAUDIO_SINK_CALLBACKS_H

#include "common.h"

/**
 * This class represents an interface that every sink implementation must subclass if they
 * wish to implement an Audio sink. You should look at the documentation for class
 * AudioSink for a pseudocode example of how to set up an AudioSink.
 */
class IAudioSinkCallbacks {
public:
    virtual ~IAudioSinkCallbacks() { }
    /**
     * Gets invoked when new data is available from the phone.
     * @param sessionId The session id that should be used while acknowledging this frame.
     * @param timestamp The timestamp of this video frame.
     * @param data A pointer to the data. The data must be copied if it cannot be processed
     *        immediately, this pointer may be freed after the call returns.
     * @param len The length of data in bytes.
     */
    virtual void dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
            uint8_t* data, size_t len) = 0;
    /**
     * Zero copy version of the data available callback. If you do not care about reducing
     * this copy, just leave this function alone and the default implementation will call
     * the non-zero copy version of the function. If you do override it, you are free to
     * mangle the GAL frame as you please, that IoBuffer will never be touched again.
     * @param sessionId The session id that should be used while acknowledging this frame.
     * @param timestamp The timestamp of this video frame.
     * @param galFrame  A shared pointer to the entire gal frame. This buffer is now yours,
     *        ownership is transferred and the receiver lib will never touch this memory
     *        again. If galFrame->raw() < data, you are free to write to that space.
     * @param data A pointer to the audio data within the galFrame.
     * @param len The length of video data.
     */
    virtual void dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
            const shared_ptr<IoBuffer>& galFrame, uint8_t* data, size_t len) {
        dataAvailableCallback(sessionId, timestamp, data, len);
    }
    /**
     * Gets invoked when there is new codec configuration data.
     * @param data A pointer to the new configuration data.
     * @param len Length of the data in bytes.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    virtual int codecConfigCallback(uint8_t* data, size_t len) = 0;
    /**
     * Called when a setup message is received.
     * @param mediaCodecType Can be one of MEDIA_CODEC_AUDIO_PCM, MEDIA_CODEC_AUDIO_AAC_LC.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    virtual int setupCallback(int mediaCodecType) = 0;
    /**
     * Called when playback is going to start. The implementation should be fully prepared
     * to accept video frames once this method returns.
     * @param sessionId The session id associated with this playback.
     */
    virtual void playbackStartCallback(int32_t sessionId) = 0;
    /**
     * Called when playback of this session has ended.
     * @param sessionId The session that is stopping.
     */
    virtual void playbackStopCallback(int32_t sessionId) = 0;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_IAUDIO_SINK_CALLBACKS_H
